package lrs;

/**
 * Represents the abstract list state. Has a concrete toString () method that
 * uses anynomous inner classes to compute the String representation of the
 * LRStruct owner.
 * @author Dung X. Nguyen and Stephen Wong  Copyright 2005 - All rights reserved.
 * @since 8/25/05
 * @stereotype abstract state
 */
abstract class ANode<E> {
  /**
   * Uses anonymous visitor class to compute a String representation.
   */
  public static final IAlgo<Object, String, Object> ToStringAlgo = new IAlgo<Object, String, Object>() {
    private final IAlgo<Object, String, Object> Helper = new IAlgo<Object, String, Object>() {
      public String emptyCase(LRStruct<? extends Object> h, Object... i) {
        return ")";
      }
      
      public String nonEmptyCase(LRStruct<? extends Object> h, Object... i) {
        return " " + h.getFirst() + h.getRest().execute (this);
      }
    };
    
    public String emptyCase(LRStruct<? extends Object> host, Object... inp) {
      return "()";
    }

    public String nonEmptyCase(LRStruct<? extends Object> host, Object... inp) {
      return "(" + host.getFirst() + host.getRest().execute(Helper);
    }
  };
  
  String toString(LRStruct<E> owner) {
    return owner.execute(ToStringAlgo);
  }
  
  /**
   * Returns the tail LRStruct of the referencing LRStruct.
   * @param owner the LRStruct referencing this ANode.
   * @return the tail LRStruct of owner.
   * @throw java.util.NoSuchElementException if empty.
   */
  abstract LRStruct<E> getRest(LRStruct<E> owner);
  
  /**
   * Returns the first data object of the referencing LRStruct.
   * @param owner the LRStruct referencing this ANode.
   * @return the tail LRStruct of owner.
   * @throw java.util.NoSuchElementException if empty.
   */
  abstract E getFirst(LRStruct<E> owner);
  
  /**
   * Sets a new tail for the referencing LRStruct.
   * @param tail the new tail for the owner LRStruct.
   * @param owner the LRS referencing this ANode.
   * @throw java.util.NoSuchElementException if empty.
   * @return <code>LRStruct</code> owner
   */
  abstract LRStruct<E> setRest(LRStruct<E> tail, LRStruct<E> owner);
  
  /**
   * Sets a new first data object for the referencing LRStruct.
   * @param first the new data object for this ANode.
   * @param owner the LRS referencing this ANode.
   * @throw java.util.NoSuchElementException if empty.
   * @return <code>LRStruct</code> owner
   */
  abstract LRStruct<E> setFirst(E dat, LRStruct<E> owner);
  
  /**
   * Inserts a data object at the front of the LRStruct owner.
   * @param dat the object to be inserted at the front.
   * @param owner the LRS referencing this ANode.
   * @return <code>LRStruct</code> owner
   */
  abstract LRStruct<E> insertFront(E dat, LRStruct<E> owner);
  
  /**
   * Removes and returns the first data object for the referencing LRStruct.
   * @param owner the LRS referencing this ANode.
   * @return the front data of the LRStruct owner.
   */
  abstract E removeFront(LRStruct<E> owner);
  
  /**
   * Executes a visitor algorithm and returns the output.
   * @param owner the LRStruct referencing this ANode.
   * @param algo the visitor algorithm to be executed.
   * @param inp the inputs needed by the algorithm.
   */
  abstract <R, P> R execute(LRStruct<E> owner, IAlgo<? super E, R, P> algo, P... inp);
}

